<?php
// Public Invoice Page - Accessible via temporary link
session_start();
include 'includes/db.php';

// Get invoice ID and token from URL
$invoice_id = isset($_GET['id']) ? (int)$_GET['id'] : 0;
$token = isset($_GET['token']) ? $_GET['token'] : '';

if (!$invoice_id || !$token) {
    die('Invalid link');
}

// Verify token and check if link is still valid (30 days)
$token_query = "SELECT * FROM invoice_links WHERE invoice_id = ? AND token = ? AND created_at >= DATE_SUB(NOW(), INTERVAL 30 DAY) AND is_active = 1";
$stmt = $conn->prepare($token_query);
$stmt->bind_param('is', $invoice_id, $token);
$stmt->execute();
$token_result = $stmt->get_result();

if ($token_result->num_rows === 0) {
    die('Link has expired or is invalid');
}

$token_data = $token_result->fetch_assoc();

// Get order details
$order_query = "SELECT o.*, c.name as customer_name, c.phone as customer_phone, c.address as customer_address 
                FROM orders o 
                LEFT JOIN customers c ON o.customer_id = c.id 
                WHERE o.id = ?";
$stmt = $conn->prepare($order_query);
$stmt->bind_param('i', $invoice_id);
$stmt->execute();
$order_result = $stmt->get_result();

if ($order_result->num_rows === 0) {
    die('Invoice not found');
}

$order = $order_result->fetch_assoc();

// Get shop info
$shop_query = "SELECT * FROM shop_info WHERE id = 1";
$shop_result = $conn->query($shop_query);
$shop = $shop_result->fetch_assoc();

// Get prices
$prices = [];
$prices_result = $conn->query("SELECT * FROM prices");
while ($price = $prices_result->fetch_assoc()) {
    $prices[$price['item_type']] = $price;
}

// Function to generate item row
function item_row($bil, $label, $qty, $price) {
    $total = $qty * $price;
    return "<tr><td>$bil</td><td>$label</td><td>$qty</td><td>RM ".number_format($price,2)."</td><td>RM ".number_format($total,2)."</td></tr>";
}
?>

<!DOCTYPE html>
<html lang="ms">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Invoice #<?= $invoice_id ?> - <?= htmlspecialchars($shop['shop_name']) ?></title>
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css">
    <style>
        body { 
            background: #f8f9fa; 
            font-family: Arial, sans-serif;
        }
        .invoice-box {
            background: #fff;
            padding: 32px 40px;
            border-radius: 12px;
            box-shadow: 0 2px 16px rgba(0,0,0,0.07);
            margin: 40px auto;
            max-width: 700px;
        }
        
        /* Print styles for A4 */
        @media print {
            @page {
                size: A4;
                margin: 15mm;
            }
            
            body {
                background: white !important;
                margin: 0;
                padding: 0;
            }
            
            .invoice-box {
                box-shadow: none;
                border: none;
                margin: 0;
                padding: 0;
                max-width: none;
                border-radius: 0;
            }
            
            .no-print {
                display: none !important;
            }
        }
        
        .invoice-header { 
            border-bottom: 2px solid #eee; 
            margin-bottom: 24px; 
            padding-bottom: 16px; 
        }
        .invoice-title { 
            font-size: 1.5rem; 
            font-weight: 700; 
        }
        .invoice-no { 
            font-size: 1.1rem; 
            color: #888; 
            text-align: right; 
        }
        .bank-box { 
            background: #f1f3f6; 
            border-radius: 8px; 
            padding: 12px 18px; 
            margin-top: 18px; 
        }
        .signature-box { 
            margin-top: 48px; 
            text-align: right; 
            display: flex;
            flex-direction: column;
            align-items: flex-end;
        }
        .signature-img { 
            max-width: 180px; 
            display: block; 
            margin-bottom: -5px; 
        }
        
        .stamp-box {
            margin-top: 48px;
            text-align: left;
        }
        
        .stamp-img {
            max-width: 120px;
            display: block;
        }
        
        /* Table styling */
        .table th { 
            background: #343a40 !important; 
            color: white !important;
            border-top: none; 
            font-weight: 600; 
        }
        
        .table-dark th {
            background: #212529 !important;
            color: white !important;
        }
        
        .table-light th {
            background: #6c757d !important;
            color: white !important;
        }
        
        /* Text alignment for specific columns */
        .table th:nth-child(3),
        .table th:nth-child(4),
        .table th:nth-child(5),
        .table td:nth-child(3),
        .table td:nth-child(4),
        .table td:nth-child(5),
        .table-dark th:nth-child(5),
        .table-dark th:last-child {
            text-align: right !important;
        }
        
        /* Mobile responsive */
        @media (max-width: 768px) {
            .invoice-box {
                margin: 20px 15px;
                padding: 20px;
            }
            
            .invoice-header .col-2,
            .invoice-header .col-8 {
                text-align: center;
                margin-bottom: 15px;
            }
            
            .invoice-no {
                text-align: center !important;
            }
        }
    </style>
</head>
<body>
<div class="invoice-box">
    <div class="row invoice-header align-items-center">
        <div class="col-2">
            <?php if ($shop['logo_image']): ?>
                <img src="/eoderalh/uploads/<?= htmlspecialchars($shop['logo_image']) ?>" alt="Logo" style="max-width:180px;">
            <?php endif; ?>
        </div>
        <div class="col-8 text-center">
            <div class="invoice-title mb-1"><?= htmlspecialchars($shop['shop_name']) ?></div>
            <div style="font-size: 0.9rem; margin-bottom: 2px;"><?= nl2br(htmlspecialchars($shop['shop_address'])) ?></div>
            <div style="font-size: 0.9rem; margin-bottom: 0;">No Tel: <?= htmlspecialchars($shop['shop_phone']) ?></div>
        </div>
        <div class="col-2 invoice-no">
            <strong>INVOICE #<?= $invoice_id ?></strong>
            <?php if ($order['status_bayar'] === 'Sudah'): ?>
                <div style="margin-top: 10px;">
                    <span style="background: #28a745; color: white; padding: 5px 10px; border-radius: 20px; font-size: 0.8rem; font-weight: bold;">PAID</span>
                </div>
            <?php endif; ?>
        </div>
    </div>
    
    <div class="row mb-3">
        <div class="col-6">
            <strong>Nama:</strong> <?= htmlspecialchars($order['customer_name']) ?><br>
            <strong>No Tel:</strong> <?= htmlspecialchars($order['customer_phone']) ?><br>
            <strong>Alamat:</strong> <?= nl2br(htmlspecialchars($order['customer_address'])) ?>
            <?php if ($order['perkara']): ?>
                <br><strong>Perkara:</strong> <?= nl2br(htmlspecialchars($order['perkara'])) ?>
            <?php endif; ?>
        </div>
        <div class="col-6 text-end">
            <strong>Tarikh Order:</strong> <?= date('d/m/Y', strtotime($order['order_date'])) ?>
            <?php if ($order['due_date']): ?>
                <br><strong>Tarikh Siap:</strong> <?= date('d/m/Y', strtotime($order['due_date'])) ?>
            <?php endif; ?>
        </div>
    </div>
    
    <table class="table table-bordered align-middle mb-4">
        <thead class="table-dark">
            <tr>
                <th>BIL</th>
                <th>Jenis</th>
                <th>Kuantiti</th>
                <th>Harga Seunit</th>
                <th>Jumlah</th>
            </tr>
        </thead>
        <tbody>
            <?php 
            $bil = 1;
            if ($order['collar_ss'] > 0) {
                echo item_row($bil++, 'Kolar SS', $order['collar_ss'], $prices['collar_ss']['sale_price']);
            }
            if ($order['collar_ls'] > 0) {
                echo item_row($bil++, 'Kolar LS', $order['collar_ls'], $prices['collar_ls']['sale_price']);
            }
            if ($order['rn_ss'] > 0) {
                echo item_row($bil++, 'RN SS', $order['rn_ss'], $prices['rn_ss']['sale_price']);
            }
            if ($order['rn_ls'] > 0) {
                echo item_row($bil++, 'RN LS', $order['rn_ls'], $prices['rn_ls']['sale_price']);
            }
            if ($order['muslimah'] > 0) {
                echo item_row($bil++, 'Muslimah', $order['muslimah'], $prices['muslimah']['sale_price']);
            }
            if ($order['nama_nombor'] > 0) {
                echo item_row($bil++, 'Nama/Nombor', $order['nama_nombor'], $prices['nama_nombor']['sale_price']);
            }
            if ($order['kain_diamond'] > 0) {
                echo item_row($bil++, 'Kain Diamond', $order['kain_diamond'], $prices['kain_diamond']['sale_price']);
            }
            ?>
            <tr>
                <td><?= $bil ?></td>
                <td>Kos Pos</td>
                <td>-</td>
                <td>-</td>
                <td>RM <?= number_format($order['shipping_cost'],2) ?></td>
            </tr>
            <?php if (isset($order['discount']) && $order['discount'] > 0): ?>
            <tr>
                <td><?= $bil + 1 ?></td>
                <td>Diskaun</td>
                <td>-</td>
                <td>-</td>
                <td>-RM <?= number_format($order['discount'],2) ?></td>
            </tr>
            <?php endif; ?>
            <tr class="table-dark">
                <th colspan="4">Jumlah Keseluruhan</th>
                <th>RM <?= number_format($order['total_sale'],2) ?></th>
            </tr>
        </tbody>
    </table>
    
    <div class="bank-box">
        <strong>Pembayaran boleh dibuat ke:</strong><br>
        <?= htmlspecialchars($shop['bank_name']) ?> (<?= htmlspecialchars($shop['bank_account']) ?>)<br>
        <strong>Nama Pemilik:</strong> <?= htmlspecialchars($shop['bank_owner']) ?>
        
        <?php if ($order['status_bayar'] === 'Sudah' && $order['payment_method']): ?>
            <hr style="margin: 10px 0; border-color: #dee2e6;">
            <strong>Maklumat Pembayaran:</strong><br>
            <strong>Status:</strong> <span style="color: #28a745; font-weight: bold;">SUDAH BAYAR</span><br>
            <strong>Kaedah:</strong> <?= htmlspecialchars($order['payment_method']) ?><br>
            <?php if ($order['payment_date']): ?>
                <strong>Tarikh Bayaran:</strong> <?= date('d/m/Y', strtotime($order['payment_date'])) ?>
            <?php endif; ?>
        <?php endif; ?>
    </div>
    
    <div class="row">
        <div class="col-6">
            <div class="stamp-box">
                <?php if ($shop['stamp_image']): ?>
                    <img src="/eoderalh/uploads/<?= htmlspecialchars($shop['stamp_image']) ?>" class="stamp-img" alt="Cop">
                <?php endif; ?>
            </div>
        </div>
        <div class="col-6">
            <div class="signature-box">
                <?php if ($shop['sign_image']): ?>
                    <img src="/eoderalh/uploads/<?= htmlspecialchars($shop['sign_image']) ?>" class="signature-img" alt="Tandatangan">
                <?php endif; ?>
                <div><strong><?= htmlspecialchars($shop['manager_name']) ?></strong></div>
            </div>
        </div>
    </div>
    
    <div class="mt-4 text-center">
        <small class="text-muted">Link ini sah selama 30 hari sahaja.</small>
    </div>
</div>

<div class="text-center mt-3 mb-3 no-print">
    <button onclick="window.print()" class="btn btn-primary">
        <i class="bi bi-printer"></i> Cetak Invoice
    </button>
    <a href="index.php" class="btn btn-secondary">
        <i class="bi bi-arrow-left"></i> Kembali ke Sistem
    </a>
</div>

<script>
// Auto print when page loads (optional - uncomment if needed)
// window.onload = function() {
//     window.print();
// };
</script>
</body>
</html> 